% // ====================================================================
% // This file is part of the Endmember Induction Algorithms Toolbox for MATLAB 
% // Copyright (C) Grupo de Inteligencia Computacional, Universidad del 
% // País Vasco (UPV/EHU), Spain, released under the terms of the GNU 
% // General Public License.
% //
% // Endmember Induction Algorithms Toolbox is free software: you can redistribute 
% // it and/or modify it under the terms of the GNU General Public License 
% // as published by the Free Software Foundation, either version 3 of the 
% // License, or (at your option) any later version.
% //
% // Endmember Induction Algorithms Toolbox is distributed in the hope that it will
% // be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
% // of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
% // General Public License for more details.
% //
% // You should have received a copy of the GNU General Public License
% // along with Endmember Induction Algorithms Toolbox. 
% // If not, see <http://www.gnu.org/licenses/>.
% // ====================================================================
%% [W,M] = EIA_LAM(X,Y)
% 
% Manuel Grana <manuel.grana[AT]ehu.es>
% Miguel Angel Veganzones <miguelangel.veganzones[AT]ehu.es>
% Grupo de Inteligencia Computacional (GIC), Universidad del Pais Vasco /
% Euskal Herriko Unibertsitatea (UPV/EHU)
% http://www.ehu.es/computationalintelligence
% 
% Copyright (2011) Grupo de Inteligencia Computacional @ Universidad del Pais Vasco, Spain.
%
% The Lattice Associative Memories is a kind of associative memory working
% over lattice operations. If X=Y then W and M are Lattice AutoAssociative
% Memories (LAAM), otherwise they are Lattice HeteroAssociative Memories
% (LHAM).
% ------------------------------------------------------------------------------
% Input:   X         : input pattern matrix [nvariables x nsamples]
%          Y         : output pattern matrix [mvariables x nsamples]
%
% Output:  W         : dilative LAM [mvariables x nvariables]
%          M         : erosive LAM [mvariables x nvariables]
%
% Bibliographical references:
% [1] M. Graña, D. Chyzhyk, M. García-Sebastián, y C. Hernández, “Lattice independent component analysis for functional magnetic resonance imaging”, Information Sciences, vol. 181, nº. 10, págs. 1910-1928, May. 2011.
function [W,M] = EIA_LAM(X,Y)

%% Check parameters
if (nargin < 1)
    error('Insufficient parameters');
end
if (nargin < 2)
    Y = X;
end
% Checking data size
[nvariables kx] = size(X);
[mvariables ky] = size(Y);
if (kx ~= ky) || kx == 0
    error('Input and output dimensions missmatch or zero dimensionality');
end

%% Compute vector lattice external products
W = zeros(mvariables,nvariables);
M = zeros(mvariables,nvariables);
for b = 1:mvariables
    for c = 1:nvariables
        product = Y(b,:) - X(c,:);
        W(b,c) = min(product);
        M(b,c) = max(product);
    end
end
